package jcircus;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import jcircus.exceptions.FailParsingException;
import jcircus.exceptions.FailTranslationException;
import jcircus.exceptions.FailTypeCheckingException;
import jcircus.translator.Translator2Java;
import jcircus.gui.JCircusFrame;
import jcircus.util.ProcInfo;
import net.sourceforge.czt.session.Markup;
import net.sourceforge.czt.session.SectionManager;
import net.sourceforge.czt.typecheck.circus.TypeChecker;
import net.sourceforge.czt.z.ast.Spec;
import circparser.CircusParser;
import net.sourceforge.czt.z.ast.Type;
import net.sourceforge.czt.z.impl.TypeImpl;

/**
 * JCircus.java
 *
 * Main class of JCircus, with access to all the modules of the tool.
 *
 * @author Angela Freitas
 *
 */
public class JCircus {
    
    /* GUI */
    private JCircusFrame jCircusFrame_;
    
    /* Translator */
    private Translator2Java translator_;
    
    /* True if it is being used with the graphical interface */
    private boolean useWithGUI_;
    
    /**
     * Constructor
     */
    public JCircus() {
        this.useWithGUI_ = false;
    }
         
    /**
     * Constructor
     *
     * @param   JCircusFrame    Reference to the GUI.
     */
    public JCircus(JCircusFrame jCircusFrame) {
        this.jCircusFrame_ = jCircusFrame;
        this.useWithGUI_ = true;
    }

    /**
     * Return the translator.
     */
    public Translator2Java getTranslator2Java() {
        return this.translator_;
    }
    
    /**
     * Performs parser, type checking and translates the given input file.
     *
     */
    public void jCircusMain(String path, String projectName, 
            String projectDir) throws Exception {
        
        // Parsing
        Spec spec = CircusParser.parse(path);
        // TODO: Change this. The errors should be a simple list, as it is in the type
        // checker, and not a mapping.
        Map<String, List<String>> errors = CircusParser.getErrors();

        if (errors.isEmpty()) {
            if (jCircusFrame_ != null)
                jCircusFrame_.reportParsingOK();
            else
                System.out.println("No parse errors, good!");
            
        } else {
            // Parsing errors
            List<String> parseErrors = null;
            
            int c = 0;
            for (List<String> l : errors.values()) {
                parseErrors = l;
                c++;
                // This should not happen because the mapping must have only one
                // value
                if (c > 1) throw new RuntimeException();
            }
            
            System.err.println(parseErrors.size() + " error(s) in total, bad!");
            for(String e : parseErrors)
                System.err.println("\t" + e);

            throw new FailParsingException(parseErrors);
        }

        //Typechecking
        SectionManager manager = new SectionManager();
        TypeChecker typeChecker = new TypeChecker(manager, Markup.LATEX);
        typeChecker.visitTerm(spec);

        List typeErrors = typeChecker.errors();

        if (typeErrors.isEmpty()) {
            
            if (jCircusFrame_ != null)
                jCircusFrame_.reportTypeCheckingOK();
            else
                System.out.println("No typing errors, good!");
            
        } else {
            System.out.println(typeErrors.size() + " type error(s) in total, bad!");            
            for(int i=0; i<typeErrors.size(); i++) {
                System.out.println(typeErrors.get(i));
            }                
            throw new FailTypeCheckingException(typeErrors);
        }    

        // Translation
        try {
            translator_ = new Translator2Java(projectDir, projectName, spec);
            
            // If we are not using the GUI, we will create a main class
            // for every process that is not parameterized
            translator_.translate(useWithGUI_);

            if (useWithGUI_) {

                // If we are using the GUI, JCircus will prompt for the processes
                // for each we want a main and the parameters
                List<ProcInfo> procInfoList = translator_.getProcInfoList();
                boolean confirmTranslation = jCircusFrame_.promptForMainProcesses(procInfoList);

                if (!confirmTranslation) {

                    return;
                }

                for (int i = 0; i < procInfoList.size(); i++) {
                    ProcInfo procInfo = procInfoList.get(i);

                    if (procInfo.getCreateMain() == true &&
                            procInfo.getParameters() != null &&
                            !procInfo.getParameters().isEmpty()) {

                        confirmTranslation = jCircusFrame_.promptForParameters(procInfo);

                        if (!confirmTranslation)
                            return;
                    }
                }

                translator_.createSources(); 
            }
            
        } catch (FailTranslationException fte) {
            
            // Problem with the input file
            List transErrors = fte.getErrors();
            
            System.out.println(transErrors.size() + " problems(s) in the input file!");            
            for(int i=0; i<transErrors.size(); i++) {
                System.out.println(transErrors.get(i));
            }        
            
            // rethrow the exception
            throw fte;
        }
        
        // No problem at all!!
        if (jCircusFrame_ != null)
            jCircusFrame_.reportTranslationOK(projectDir, projectName);
        else
            System.out.println("Translation sucessfull!");
    }
    
    /**
     * Main method.
     */
    public static void main(String[] args) {

        try {
            if (args.length < 3) {
                System.out.println("Error: Wrong number of parameters");
                System.out.println("Use: java jCircus.Jcircus <input-file> <proj-name> <output-dir>");
                System.exit(0);
            }

            String path = args[0];
            String projectName = args[1]; 
            String projectDir = args[2]; 
            String nameMainProcess = null;

            if (args.length > 3) {
                nameMainProcess = args[3];
            }

            JCircus jCircus = new JCircus();
            jCircus.jCircusMain(path, projectName, projectDir);
            
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
    
}
